/*
 * Telecom Clock driver for Wainwright board
 *
 * Copyright (C) 2005 Kontron Canada
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 * NON INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send feedback to <sebastien.bouchard@ca.kontron.com>
 *
 * Description : This is the TELECOM CLOCK module driver for the ATCA platform.
 */
#ifndef __KERNEL__
#  define __KERNEL__
#endif

#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/kernel.h>	/* printk() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/delay.h>	/* udelay */
#include <asm/uaccess.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <linux/timer.h>
#include <linux/miscdevice.h>
#include <asm/io.h>		/* inb/outb */
#include "tlclk.h"	/* TELECOM IOCTL DEFINE */

MODULE_AUTHOR("Sebastien Bouchard <sebastien.bouchard@ca.kontron.com>");
MODULE_LICENSE("GPL");

/* Telecom clock I/O register definition */
#define TLCLK_BASE 0xa08
#define TLCLK_REG0 TLCLK_BASE
#define TLCLK_REG1 (TLCLK_BASE+1)
#define TLCLK_REG2 (TLCLK_BASE+2)
#define TLCLK_REG3 (TLCLK_BASE+3)
#define TLCLK_REG4 (TLCLK_BASE+4)
#define TLCLK_REG5 (TLCLK_BASE+5)
#define TLCLK_REG6 (TLCLK_BASE+6)
#define TLCLK_REG7 (TLCLK_BASE+7)

#define SET_PORT_BITS(port, mask, val) outb(((inb(port) & mask) | val), port)

/* 0 = Dynamic allocation of the major device number */
#define TLCLK_MAJOR 0

/* Contain the interrupt used for telecom clock */
static unsigned int telclk_interrupt;

static int int_events;		/* Event that generate a interrupt */
static int got_event;		/* if events processing have been done */

static void switchover_timeout(unsigned long data);
static struct timer_list switchover_timer =
	TIMER_INITIALIZER(switchover_timeout , 0, 0);

static struct tlclk_alarms *alarm_events;

static DEFINE_SPINLOCK(event_lock);

static int tlclk_major = TLCLK_MAJOR;

static irqreturn_t tlclk_interrupt(int irq, void *dev_id, struct pt_regs *regs);

static DECLARE_WAIT_QUEUE_HEAD(wq);
/*
*  Function : Module I/O functions
*  Description : Almost all the control stuff is done here, check I/O dn for
*  help.
*/
static int
tlclk_ioctl(struct inode *inode,
	    struct file *filp, unsigned int cmd, unsigned long arg)
{
	unsigned long flags;
	unsigned char val;
	val = (unsigned char) arg;

	if (_IOC_TYPE(cmd) != TLCLK_IOC_MAGIC)
		return -ENOTTY;

	if (_IOC_NR(cmd) > TLCLK_IOC_MAXNR)
		return -ENOTTY;

	switch (cmd) {
	case IOCTL_RESET:
		SET_PORT_BITS(TLCLK_REG4, 0xfd, val);
		break;
	case IOCTL_MODE_SELECT:
		SET_PORT_BITS(TLCLK_REG0, 0xcf, val);
		break;
	case IOCTL_REFALIGN:
		/* GENERATING 0 to 1 transistion */
		SET_PORT_BITS(TLCLK_REG0, 0xf7, 0);
		udelay(2);
		SET_PORT_BITS(TLCLK_REG0, 0xf7, 0x08);
		udelay(2);
		SET_PORT_BITS(TLCLK_REG0, 0xf7, 0);
		break;
	case IOCTL_HARDWARE_SWITCHING:
		SET_PORT_BITS(TLCLK_REG0, 0x7f, val);
		break;
	case IOCTL_HARDWARE_SWITCHING_MODE:
		SET_PORT_BITS(TLCLK_REG0, 0xbf, val);
		break;
	case IOCTL_FILTER_SELECT:
		SET_PORT_BITS(TLCLK_REG0, 0xfb, val);
		break;
	case IOCTL_SELECT_REF_FREQUENCY:
		spin_lock_irqsave(&event_lock, flags);
		SET_PORT_BITS(TLCLK_REG1, 0xfd, val);
		spin_unlock_irqrestore(&event_lock, flags);
		break;
	case IOCTL_SELECT_REDUNDANT_CLOCK:
		spin_lock_irqsave(&event_lock, flags);
		SET_PORT_BITS(TLCLK_REG1, 0xfe, val);
		spin_unlock_irqrestore(&event_lock, flags);
		break;
	case IOCTL_SELECT_AMCB1_TRANSMIT_CLOCK:
		if ((val == CLK_8kHz) || (val == CLK_16_384MHz)) {
			SET_PORT_BITS(TLCLK_REG3, 0xf8, 0x5);
			SET_PORT_BITS(TLCLK_REG1, 0xfb, ~val);
		} else if (val >= CLK_8_592MHz) {
			SET_PORT_BITS(TLCLK_REG3, 0xf8, 0x7);
			switch (val) {
			case CLK_8_592MHz:
				SET_PORT_BITS(TLCLK_REG0, 0xfc, 1);
				break;
			case CLK_11_184MHz:
				SET_PORT_BITS(TLCLK_REG0, 0xfc, 0);
				break;
			case CLK_34_368MHz:
				SET_PORT_BITS(TLCLK_REG0, 0xfc, 3);
				break;
			case CLK_44_736MHz:
				SET_PORT_BITS(TLCLK_REG0, 0xfc, 2);
				break;
			}
		} else
			SET_PORT_BITS(TLCLK_REG3, 0xf8, val);
		break;
	case IOCTL_SELECT_AMCB2_TRANSMIT_CLOCK:
		if ((val == CLK_8kHz) || (val == CLK_16_384MHz)) {
			SET_PORT_BITS(TLCLK_REG3, 0xc7, 0x28);
			SET_PORT_BITS(TLCLK_REG1, 0xfb, ~val);
		} else if (val >= CLK_8_592MHz) {
			SET_PORT_BITS(TLCLK_REG3, 0xc7, 0x38);
			switch (val) {
			case CLK_8_592MHz:
				SET_PORT_BITS(TLCLK_REG0, 0xfc, 1);
				break;
			case CLK_11_184MHz:
				SET_PORT_BITS(TLCLK_REG0, 0xfc, 0);
				break;
			case CLK_34_368MHz:
				SET_PORT_BITS(TLCLK_REG0, 0xfc, 3);
				break;
			case CLK_44_736MHz:
				SET_PORT_BITS(TLCLK_REG0, 0xfc, 2);
				break;
			}
		} else
			SET_PORT_BITS(TLCLK_REG3, 0xc7, val << 3);
		break;
	case IOCTL_TEST_MODE:
		SET_PORT_BITS(TLCLK_REG4, 0xfd, 2);
		break;
	case IOCTL_ENABLE_CLKA0_OUTPUT:
		SET_PORT_BITS(TLCLK_REG2, 0xfe, val);
		break;
	case IOCTL_ENABLE_CLKB0_OUTPUT:
		SET_PORT_BITS(TLCLK_REG2, 0xfd, val << 1);
		break;
	case IOCTL_ENABLE_CLKA1_OUTPUT:
		SET_PORT_BITS(TLCLK_REG2, 0xfb, val << 2);
		break;
	case IOCTL_ENABLE_CLKB1_OUTPUT:
		SET_PORT_BITS(TLCLK_REG2, 0xf7, val << 3);
		break;
	case IOCTL_ENABLE_CLK3A_OUTPUT:
		SET_PORT_BITS(TLCLK_REG3, 0xbf, val << 6);
		break;
	case IOCTL_ENABLE_CLK3B_OUTPUT:
		SET_PORT_BITS(TLCLK_REG3, 0x7f, val << 7);
		break;
	case IOCTL_READ_ALARMS:
		return (inb(TLCLK_REG2) & 0xf0);
		break;
	case IOCTL_READ_INTERRUPT_SWITCH:
		return inb(TLCLK_REG6);
		break;
	case IOCTL_READ_CURRENT_REF:
		return ((inb(TLCLK_REG1) & 0x08) >> 3);
		break;
	}

	return 0;
}


static int tlclk_open(struct inode *inode, struct file *filp)
{
	int result;

	/* Make sure there is no interrupt pending while
	 * initialising interrupt handler */
	inb(TLCLK_REG6);

	/* This device is wired through the FPGA IO space of the ATCA blade
	 * we can't share this IRQ */
	result = request_irq(telclk_interrupt, &tlclk_interrupt,
			     SA_INTERRUPT, "telco_clock", tlclk_interrupt);
	if (result == -EBUSY) {
		printk(KERN_ERR "tlclk: Interrupt can't be reserved.\n");
		return -EBUSY;
	}
	inb(TLCLK_REG6);	/* Clear interrupt events */

	return 0;
}

static int tlclk_release(struct inode *inode, struct file *filp)
{
	free_irq(telclk_interrupt, tlclk_interrupt);

	return 0;
}

static ssize_t tlclk_read(struct file *filp, char __user *buf, size_t count,
		loff_t *f_pos)
{
	if (count < sizeof(struct tlclk_alarms))
		return -EIO;

	wait_event_interruptible(wq, got_event);
	if (copy_to_user(buf, alarm_events, sizeof(struct tlclk_alarms)))
		return -EFAULT;

	memset(alarm_events, 0, sizeof(struct tlclk_alarms));
	got_event = 0;

	return  sizeof(struct tlclk_alarms);
}

static ssize_t tlclk_write(struct file *filp, const char __user *buf, size_t count,
	loff_t *f_pos)
{
	return 0;
}

static struct file_operations tlclk_fops = {
	.read = tlclk_read,
	.write = tlclk_write,
	.ioctl = tlclk_ioctl,
	.open = tlclk_open,
	.release = tlclk_release,

};

static struct miscdevice tlclk_miscdev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "telco_clock",
	.fops = &tlclk_fops,
};



static int __init tlclk_init(void)
{
	int ret;

	ret = register_chrdev(tlclk_major, "telco_clock", &tlclk_fops);
	if (ret < 0) {
		printk(KERN_ERR "tlclk: can't get major %d.\n", tlclk_major);
		return ret;
	}
	tlclk_major = ret;
	alarm_events = kzalloc( sizeof(struct tlclk_alarms), GFP_KERNEL);
	if (!alarm_events)
		goto out1;

	/* Read telecom clock IRQ number (Set by BIOS) */
	if (!request_region(TLCLK_BASE, 8, "telco_clock")) {
		printk(KERN_ERR "tlclk: request_region 0x%X failed.\n",
			TLCLK_BASE);
		ret = -EBUSY;
		goto out2;
	}
	telclk_interrupt = (inb(TLCLK_REG7) & 0x0f);

	if (0x0F == telclk_interrupt ) { /* not MCPBL0010 ? */
		printk(KERN_ERR "telclk_interrup = 0x%x non-mcpbl0010 hw.\n",
			telclk_interrupt);
		ret = -ENXIO;
		goto out3;
	}

	init_timer(&switchover_timer);

	ret = misc_register(&tlclk_miscdev);
	if (ret < 0) {
		printk(KERN_ERR "tlclk: misc_register returns %d.\n", ret);
		ret = -EBUSY;
		goto out3;
	}


	return 0;
out3:
	release_region(TLCLK_BASE, 8);
out2:
	kfree(alarm_events);
out1:
	unregister_chrdev(tlclk_major, "telco_clock");
	return -EBUSY;
}

static void __exit tlclk_cleanup(void)
{
	misc_deregister(&tlclk_miscdev);
	unregister_chrdev(tlclk_major, "telco_clock");
	release_region(TLCLK_BASE, 8);
	del_timer_sync(&switchover_timer);
	kfree(alarm_events);

}

static void switchover_timeout(unsigned long data)
{
	if ((data & 1)) {
		if ((inb(TLCLK_REG1) & 0x08) != (data & 0x08))
			alarm_events->switchover_primary++;
	} else {
		if ((inb(TLCLK_REG1) & 0x08) != (data & 0x08))
			alarm_events->switchover_secondary++;
	}

	/* Alarm processing is done, wake up read task */
	del_timer(&switchover_timer);
	got_event = 1;
	wake_up(&wq);
}

static irqreturn_t tlclk_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long flags;

	spin_lock_irqsave(&event_lock, flags);
	/* Read and clear interrupt events */
	int_events = inb(TLCLK_REG6);

	/* Primary_Los changed from 0 to 1 ? */
	if (int_events & PRI_LOS_01_MASK) {
		if (inb(TLCLK_REG2) & SEC_LOST_MASK)
			alarm_events->lost_clocks++;
		else
			alarm_events->lost_primary_clock++;
	}

	/* Primary_Los changed from 1 to 0 ? */
	if (int_events & PRI_LOS_10_MASK) {
		alarm_events->primary_clock_back++;
		SET_PORT_BITS(TLCLK_REG1, 0xFE, 1);
	}
	/* Secondary_Los changed from 0 to 1 ? */
	if (int_events & SEC_LOS_01_MASK) {
		if (inb(TLCLK_REG2) & PRI_LOST_MASK)
			alarm_events->lost_clocks++;
		else
			alarm_events->lost_secondary_clock++;
	}
	/* Secondary_Los changed from 1 to 0 ? */
	if (int_events & SEC_LOS_10_MASK) {
		alarm_events->secondary_clock_back++;
		SET_PORT_BITS(TLCLK_REG1, 0xFE, 0);
	}
	if (int_events & HOLDOVER_10_MASK)
		alarm_events->pll_end_holdover++;

	if (int_events & UNLOCK_01_MASK)
		alarm_events->pll_lost_sync++;

	if (int_events & UNLOCK_10_MASK)
		alarm_events->pll_sync++;

	/* Holdover changed from 0 to 1 ? */
	if (int_events & HOLDOVER_01_MASK) {
		alarm_events->pll_holdover++;

		/* TIMEOUT in ~10ms */
		switchover_timer.expires = jiffies + msecs_to_jiffies(10);
		switchover_timer.data = inb(TLCLK_REG1);
		add_timer(&switchover_timer);
	} else {
		got_event = 1;
		wake_up(&wq);
	}
	spin_unlock_irqrestore(&event_lock, flags);

	return IRQ_HANDLED;
}


module_init(tlclk_init);
module_exit(tlclk_cleanup);
