/*
 ********************************************************************
 *
 * rieglapi.c - API for the Riegl Q140i-60 LMS LynxOS device driver
 *
 *     CSIRO Automation
 *     Queensland Centre for Advanced Technologies
 *     PO Box 883, Kenmore, QLD 4069, Australia
 *     www.cat.csiro.au/cmst
 *
 *	$Id: rieglapi.c,v 1.4 2004/02/25 17:26:37 redteam Exp $
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 ********************************************************************
 */

//static char rcsid[] = "$Id: rieglapi.c,v 1.4 2004/02/25 17:26:37 redteam Exp $";

/**
 ********************************************************************
 *
 * \file rieglapi.c
 * \brief API for the Riegl Q140i-60 LMS LynxOS device driver
 * \author Pavan Sikka
 *
 ********************************************************************
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/ppdev.h>
#include <linux/parport.h>
#include <errno.h>
#include <math.h>
#include	<stdarg.h>

#include "rieglapi.h"

// forward defines

int nread ( int fd, unsigned char * buf, int n);
static void riegl_unpack_point ( RIEGL * rp, unsigned char * pbuf, RIEGL_DATA_POINT * p);

void * rtx_error_null(char *fmt, ...);
void * rtx_error_errno_null(char *fmt, ...);
int rtx_error_errno(char *fmt, ...);
int rtx_error(char *fmt, ...);


RIEGL *
riegl_open (
        const char * devName
    )
{
    int fd;
    unsigned char buf[BUFSIZ];
    int hdrLen;
    RIEGL * rp;
    int		addr = 0;

    // open the parallel port interface
    if ((fd = open (devName, O_RDONLY)) == -1) {
        return (rtx_error_errno_null ("riegl_open: open() failed"));
    }

    fprintf(stderr, "RIPort open\n");

    // read the header length, first 4 bytes
    if (nread (fd, buf, 4) == -1) {
      fprintf(stderr, "Read failed\n");
      return (rtx_error_null ("riegl_open: nread(headerLength) failed"));
    }
    hdrLen = buf[0] + (buf[1] << 8) + (buf[2] << 16) + (buf[3] << 24);
    fprintf(stderr, "header length is %d bytes\n", hdrLen);

    // allocate a RIEGL object
    if ((rp = (RIEGL *)calloc (1, sizeof (RIEGL))) == NULL) {
        return (rtx_error_null ("riegl_open: calloc() failed"));
    }
    rp->fd = fd;

    // read the rest of the header
    if (nread (fd, &buf[4], hdrLen - 4) == -1) {
        return (rtx_error_null ("riegl_open: nread(header) failed"));
    }

    // now unpack the data into the RIEGL object, object contains
    // chars which confound alignment, the header has no padding...
    rp->header.dwSize = hdrLen;
    addr = 4;

#define	unpack(type,el)	{ rp->header.el = *((type *)(&buf[addr])); addr += sizeof(type); };
#define	unpack_s(type,el)	{ strncpy(rp->header.el, &buf[addr], sizeof(type)); addr += sizeof(type); };

    unpack( short, wBytesPerLine )
    unpack( char, bProtocolID )
    unpack( char, bHeaderID )
    unpack( short, wMeasOffset )
    unpack( short, wMeasSize )
    unpack( short, wMeasCount )
    unpack( char, bLeadInIDMain )
    unpack( short, wLeadinIDSub )
    unpack( char, bMeasIDMain )
    unpack( short, wMeasIDSub )
    unpack( char, bTrailerIDMain )
    unpack( short, wTrailerIDSub )
    unpack( char, bParameterIDMain )
    unpack( short, wParameterIDSub )
    unpack_s( char[8], 	SerialNumber )
    unpack( float, fRangeUnit )
    unpack( float, fAngleUnit )
    unpack( float, fTimerUnit )
    unpack( char, AngleID )

    return (rp);
}

int
riegl_read (
        RIEGL * rp,
	RIEGL_DATA_POINT * data,
	struct timeval *time
    )
{
    int bytesPerPoint, i;
    short	synchWord;
    unsigned char	*pp, buf[2];
    int                 slipCount = 0;

    // read the sync word and check it
    //    fprintf(stderr, "--------------\nabout to read sync word\n");
    if (nread (rp->fd, buf, 2) == -1) {
        return (rtx_error ("riegl_read: nread(header) failed"));
    }
    while (1) {
	    synchWord = buf[0] + (buf[1] << 8);
	    if (synchWord != rp->header.wBytesPerLine) {
		    /*
		fprintf(stderr, "riegl_read: synchSeq (%d) differs from "
				     "expected value (%d)", synchWord, 
				     rp->header.wBytesPerLine);
		    */
		buf[0] = buf[1];
		nread(rp->fd, &buf[1], 1);
		slipCount++;
	    } else {
	      //	    fprintf(stderr, "got good sync word, slipped %d bytes\n");
		    break;
		}
	}

    // allocate a private buffer for one Riegl 'line' plus the time stamp
    // at the start of the buffer
    if (rp->buf == NULL)
	    rp->buf = (unsigned char *)calloc(1, 
					      rp->header.wBytesPerLine
					      + sizeof(struct timeval));
    if (rp->buf == NULL)
        return rtx_error ("riegl_read: failed to allocate private buffer");

    // copy the timeval from the first sizeof(struct timeval) bytes of the 
    // riport
    if(nread(rp->fd, time, sizeof(struct timeval)) == -1) {
      return(rtx_error("riegl_read: nread(time) failed"));
    }

    // read a line of data into private buffer
    // fprintf(stderr, "about to read %d bytes\n", rp->header.wBytesPerLine);
    if (nread (rp->fd, rp->buf, 
	       rp->header.wBytesPerLine) == -1) {
        return (rtx_error ("riegl_read: nread(scan) failed"));
    }

    // read data points from the data stream
    bytesPerPoint = rp->header.wBytesPerLine / rp->header.wMeasCount;
    for (i=0, pp=rp->buf; i<rp->header.wMeasCount; i++, pp+=bytesPerPoint)
        riegl_unpack_point (rp, pp, data++);

    return (0);
}

static void
riegl_unpack_point (
        RIEGL * rp,
	unsigned char * pbuf,
	RIEGL_DATA_POINT * p
    )
{
    double angle;
    unsigned int range;
    unsigned char * buf = pbuf;

    if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_RANGE) {
        range = buf[0] + (buf[1] << 8);
        p->range = (float)(range * rp->header.fRangeUnit );
	buf += 2;
    }
    if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_AMPLITUDE) {
        p->amplitude = buf[0];
	buf += 1;
    }
    if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_QUALITY) {
        p->quality = buf[0];
	buf += 1;
    }
    if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_ANGLE) {
        angle = 2.0 * ((buf[0] + (buf[1] << 8)) % 10000) * 0.01;
	p->angle = (M_PI / 200.0) * angle - M_PI/2.0;	// convert to rads
	buf += 2;
    }
    if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_DELAY)
        p->delay = (buf[0] + (buf[1] << 8) + (buf[2] << 16)) *
	    1.0E-5;
}

int
riegl_close (
        RIEGL * rp
    )
{
    close (rp->fd);
    free (rp);
    return (0);
}

void
riegl_print_header (
        RIEGL * rp
    )
{
#define	hprint_i(el)	hprint(el, "%8u\n");
#define	hprint_f(el)	hprint(el, "%8f\n");
#define	hprint_s(el)	hprint(el, "%8s\n");
#define	hprint(el,fmt)	printf("%16s ", #el); printf(fmt, rp->header.el)
 
    printf ("--------------------------------------------------------\n");
    hprint_i(dwSize);
    hprint_i(wBytesPerLine);
    hprint_i(bProtocolID);
    hprint_i(bHeaderID);

    // main block
    hprint_i(wMeasOffset);
    hprint_i(wMeasSize);
    hprint_i(wMeasCount);
    hprint_i(bLeadInIDMain);
    hprint_i(wLeadinIDSub);
    hprint_i(bMeasIDMain);
    hprint_i(wMeasIDSub);
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_RANGE)
	    printf ("           - range\n");
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_AMPLITUDE)
	    printf ("           - amplitude\n");
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_QUALITY)
	    printf ("           - quality\n");
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_ANGLE)
	    printf ("           - angle\n");
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_DELAY)
	    printf ("           - time\n");

    hprint_i(bTrailerIDMain);
    hprint_i(wTrailerIDSub);
    hprint_i(bParameterIDMain);
    hprint_i(wParameterIDSub);

    // parameter block
    hprint_s(	SerialNumber);
    hprint_f(	fRangeUnit);
    hprint_f(	fAngleUnit);
    hprint_f(	fTimerUnit);
    hprint_i(	AngleID);
    printf ("--------------------------------------------------------\n");
}

void
riegl_print_scan (
        RIEGL * rp,
	RIEGL_DATA_POINT	*pp
    )
{
    int i;

    printf ("--------------------------------------------------------\n");
    for (i=0; i<rp->header.wMeasCount; i++) {
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_RANGE) {
	    printf ("%10.3f ", pp->range);
	}
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_AMPLITUDE) {
	    printf ("%4d ", pp->amplitude);
	}
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_QUALITY) {
	    printf ("%4d ", pp->quality);
	}
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_ANGLE) {
	    printf ("%10.6f ", pp->angle);
	}
        if (rp->header.wMeasIDSub & RIEGL_DATA_FORMAT_DELAY) {
	    printf ("%10.6f ", pp->delay);
	}
	printf ("\n");
	pp++;
    }
    printf ("--------------------------------------------------------\n");

}

int
nread (
        int fd,
	unsigned char * buf,
	int n
    )
{
    int m = n, k = 0;

    do {
        if ((k = read (fd, (char *) buf, n)) == -1)
	    return (rtx_error_errno ("rieglapi (nread): read() failed"));
	m -= k;
	buf = &(buf[n-m]);
    } while (m > 0);
    return (n);
}


// csiro compatibility routines


void *
rtx_error_null(char *fmt, ...)
{
	va_list	ap;
	va_start(ap, fmt);

	vfprintf(stderr, fmt, ap);

	return NULL;
}

void *
rtx_error_errno_null(char *fmt, ...)
{
	va_list	ap;
	va_start(ap, fmt);

	vfprintf(stderr, fmt, ap);
	perror(": ");

	return NULL;
}

int
rtx_error(char *fmt, ...)
{
	va_list	ap;
	va_start(ap, fmt);

	vfprintf(stderr, fmt, ap);

	return -1;
}
int
rtx_error_errno(char *fmt, ...)
{
	va_list	ap;
	va_start(ap, fmt);

	vfprintf(stderr, fmt, ap);
	perror(": ");

	return -1;
}
#ifdef TEST

RIEGL * riegl;
unsigned char buf[8192];

int main ()
{
    int i, fd;

    if ((riegl = riegl_open ("/dev/riport0")) == NULL) {
        perror ("open");
	exit (1);
    }
    riegl_print_header (riegl);

    for (i=0; i<10; i++) {
	RIEGL_DATA_POINT	scan[500];

        if (riegl_read (riegl, scan) == -1) {
	    perror ("read");
	    break;
	}
	riegl_print_scan (riegl, scan);
    }

    riegl_close (riegl);

    return (0);
}

#endif
