/*
 *   riport.o 
 *   Linux device driver to access Riegl LMS scanner units via the parallel port
 *   
 *   Copyright (C) 2000  Roland Schwarz
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *   
 *   The author can be reached by email: roland.schwarz@riegl.com
 */
 
/*
 * 10.07.2000 Tested for use with Kernel >= 2.2
 * 14.03.2000 First working version
 * 10.02.2000 Start of work
 */

#define MAX_RIPORT_DEVICES 2

// default settings
#define RIPORT_IO 0x378
#define RIPORT_IRQ 7
#define RIPORT_SIZE 4000

// standard and ECP port offsets
#define ECP_OFFSET 0x400
#define ECR_EXT		2
#define DCR_BASE	2
#define FIFO_EXT	0

// bit definitions for registers
#define ECR_SPP_MODE			0x00
#define ECR_ERRINT_DISABLED		0x10
#define ECR_SERVICE_INTERRUPT	0x04
#define ECR_BYTE_MODE			0x20
#define ECR_ECP_MODE			0x60
#define DCR_NOT_REVERSE_REQUEST	0x04
#define DCR_NOT_1284_ACTIVE		0x08
#define DCR_DIRECTION			0x20
#define DCR_SELECT_IN			0x08
#define ECR_FIFO_EMPTY			0x01
		
#include <linux/module.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/wait.h>
#include <linux/time.h>

#include <asm/uaccess.h>
#include <asm/io.h>

#undef PDEBUG             // undef it, just in case
#ifdef RIPORT_DEBUG
#  ifdef __KERNEL__
// This one if debugging is on, and kernel space
#    define PDEBUG(fmt, args...) printk( KERN_DEBUG "riport: " fmt, ## args)
#  else
// This one for user space
#    define PDEBUG(fmt, args...) fprintf(stderr, "riport: " fmt, ## args)
#  endif
#else
#  define PDEBUG(fmt, args...) // not debugging: nothing
#endif

//------------------------------------------------------------------------------------//

#define RPINIT 0
#define RPREAD_HEADER 1
#define RPSYNC 2
#define RPREAD 3
#define RPDUMP_TIMESTAMP 4

struct devriport {
  int io;
  int io_ext;
  int irq;
  int dma;
  int size;                     // buffer size
  unsigned char* pbuf;          // pointer to the start of the memory that
                                // stores scans from the riegl
  unsigned char* pbuf_top;      // pointer to the end of pbuf (see above)
  unsigned char* pin;           // pointer to the end of new data
  unsigned char* pout;          // pointer to the start of new data (end of
                                // old/read data)
  wait_queue_head_t qwait;
  struct inode* pinode;
  struct file* pfile;
  int usage;
  int irqinuse;

  int readState;
  short syncWord;
  int numBytesThisState;
  int bytesToRead;
  char buf[4];
  struct timeval timeStamp;
};

struct devriport* devriport_init(int major, int minor, int io, int irq, int dma, int size, int* presult);
void devriport_cleanup(struct devriport* this);
int  devriport_open(struct devriport* this);
int  devriport_release(struct devriport* this);
int  devriport_read(struct devriport* this, char* pbuf, int length);
unsigned int devriport_poll(struct devriport* this, struct poll_table_struct* ptable);
void devriport_irq(struct devriport* this, int irq, struct pt_regs* regs);
void devriport_irq_wrap(int irq, void* pv, struct pt_regs* pr)
{ devriport_irq(pv, irq, pr); }
void devriport_rx(struct devriport* this);

struct devriport* devriport_init(int major, int minor, int io, int irq, int dma, int size, int* presult)
{
	struct devriport* this;
	//int n;
	
	*presult = 0;
	this = kmalloc(sizeof(struct devriport),GFP_KERNEL);
	if (!this) { *presult = -ENOMEM; goto fail_memory;} 

	if ((*presult=check_region(io,3))<0) goto fail_io;
	if ((*presult=check_region(io+ECP_OFFSET,3))<0) goto fail_io;

	request_region(io,3,"riport");
	request_region(io+ECP_OFFSET,3,"riport");

	this->io = io;
	this->io_ext = io+ECP_OFFSET;
	this->irq = irq;
	this->dma = dma;
	this->size = size;
	this->pinode = NULL;
	this->pfile = NULL;
	this->usage = 0;

	this->readState = RPINIT;
	this->syncWord = 0;
	this->bytesToRead = 0;
	this->numBytesThisState = 0;

	init_waitqueue_head( &this->qwait );
	//	this->qwait = NULL;
	this->irqinuse = 0;
#if 0
	n = inb(this->io_ext+ECR_EXT);
	if (!((n&1) && (!(n&2)))){ *presult = -ENODEV; goto fail_dev; }
	outb(0x34,this->io_ext+ECR_EXT);
	n = inb(this->io_ext+ECR_EXT);
	if (n!=0x35) {*presult = -ENODEV; goto fail_dev; } 
#endif
	// test if ECP port (required)
	outb(0x34,this->io_ext+ECR_EXT);
	if(0x35!=inb(this->io_ext+ECR_EXT)) {
		*presult = -ENXIO;
		goto fail_dev;
	}

	printk(KERN_NOTICE "ecp: found at io=0x%x irq=%d major=%d minor=%d size=%d\n",
		io, irq, major, minor, size);	

	return this;
	
	fail_dev:
	fail_io:
		kfree(this);
	fail_memory:
		return NULL;
}

void devriport_cleanup(struct devriport* this)
{
  // release DMA region for the ECP port (?? Not sure about this)
  release_region(this->io+ECP_OFFSET,3);
  release_region(this->io,3);

  // delete the memory for the device
  kfree(this);
}

int devriport_open(struct devriport* this)
{
	int result;

	// if this port is already being used, return EBUSY
	if (this->usage) return -EBUSY;

	// try to get a handle on the desired irq.  if it's busy, pass the 
	// kernel error from request_irq along (often EBUSY)
	result = request_irq(this->irq, devriport_irq_wrap, SA_INTERRUPT,  "riport", this); 
	if (result) goto fail_irq;

	// grab this irq
	this->irqinuse = 1;

	// malloc the buffer to store reads
	this->pbuf = kmalloc(this->size, GFP_KERNEL);
	if (!this->pbuf) result = -ENOMEM;
	if (result) goto fail_memory;

	// get a handle on the end of the buffer
	this->pbuf_top = this->pbuf+this->size-1;

	// set pin and pout to the same thing (implies an empty buffer)
	this->pin = this->pbuf;
	this->pout = this->pbuf;

	// make the driver search for a sync byte. needs a valid header
	// to find the sync
	this->readState = RPINIT;
	this->syncWord = 0;
	this->bytesToRead = 0;
	this->numBytesThisState = 0;

	// set up ECP port

	// switch to compatibility mode
	outb(ECR_SPP_MODE|ECR_ERRINT_DISABLED|ECR_SERVICE_INTERRUPT, 
	     this->io_ext+ECR_EXT);
	outb(DCR_NOT_REVERSE_REQUEST|DCR_NOT_1284_ACTIVE, 
	     this->io+DCR_BASE);

	// switch to reverse direction & disable IRQ'S
	outb(ECR_BYTE_MODE|ECR_ERRINT_DISABLED|ECR_SERVICE_INTERRUPT, 
	     this->io_ext+ECR_EXT);
	outb(DCR_DIRECTION|DCR_NOT_REVERSE_REQUEST, 
	     this->io+DCR_BASE);
	outb(ECR_ECP_MODE|ECR_ERRINT_DISABLED|ECR_SERVICE_INTERRUPT, 
	     this->io_ext+ECR_EXT);
	outb(DCR_DIRECTION, 
	     this->io+DCR_BASE);

	// increment usage... note that usage should never be greater than 1
	this->usage++;
	PDEBUG("open\n");

	// do an initial read from the riegl -- reads the header
	devriport_rx(this);

	return 0;

 fail_memory:
	// if memory couldn't be malloc'd, let go of the irq
	this->irqinuse = 0;
	free_irq(this->irq, this);

 fail_irq:
	// return the reason the open failed
	return result;
}

int devriport_release(struct devriport* this)
{
  // let go of the irq
  this->irqinuse = 0;
  
  // switch to compatibility mode
  outb(ECR_SPP_MODE|ECR_ERRINT_DISABLED|ECR_SERVICE_INTERRUPT, 
       this->io_ext+ECR_EXT);
  outb(DCR_NOT_REVERSE_REQUEST|DCR_SELECT_IN, 
       this->io+DCR_BASE);
  
  // release resource
  free_irq(this->irq, this);
  kfree(this->pbuf);
  
  // free the riport (usage should never be less than zero)
  this->usage--;
  PDEBUG("release\n");
  return 0;
}

// reads length bytes from the riport buffer to pbuf
// prepends a timeval to the data it sends back
int devriport_read(struct devriport* this, char* pbuf, int length)
{
  //printk("Reading attempt of %d\n",length);
  //  declare the wait queue, 2.4 kernel style
  DECLARE_WAITQUEUE( wait, current);

  int retval;
  int length0, length1;
  //PDEBUG("read want: %d\n",length);


  // sleep until data available
#if 0
  while(this->pin == this->pout) {
    if (this->pfile->f_flags & O_NONBLOCK)
      return -EAGAIN;
    cli();
    if (this->pin == this->pout)
      interruptible_sleep_on(&this->qwait);
    sti();
    if (signal_pending(current))
      return -ERESTARTSYS;
  }
#endif

  // defer to other threads while data is unavailible
  add_wait_queue(&this->qwait, &wait);

  // clear return value and allow the module to be interrupted (for now)
  retval = 0;
  current->state = TASK_INTERRUPTIBLE;

  // block until data arrives
  // pin will be greater than pout if new data has come in, but hasn't been 
  // read.  pin should equal pout if no data is left
  while(this->pin == this->pout) {
    // if nonblocking, return with EAGAIN (to tell the caller to try again)
    if (this->pfile->f_flags & O_NONBLOCK) {
       printk("EAGAIN Error\n");
      retval = -EAGAIN;
      break;
    }
    // defer to the scheduler voluntarily
    schedule();
    // return restart if we've been interrupted by a signal (like SIG_INT)
    if (signal_pending(current)) {
      retval = -ERESTARTSYS;
      break;
    }
  }

  // mark this process as runnable again and start running
  current->state = TASK_RUNNING;
  remove_wait_queue(&this->qwait, &wait);

  // make sure we haven't encountered an error already
  if (retval){
    return retval;
    //printk("Nothing read\n");
  }

  // read from buffer
  length0 = this->pin - this->pout;
  // the buffer is circular, so pin can be less than pout, if this is the case
  // read from pout and wrap around
  if (length0 < 0) {
    // make sure length0 reflects the actual number of bytes being read
    // (see wraparound comments above)
    length0 += this->size;

    // if length0 < length, not enough bytes are in the buffer so read only 
    // length0 bytes back to the user
    length = (length0 < length)?length0:length;

    // length1 is the number of bytes from the current read position in the
    // circular buffer to the end of the buffer
    length1 = this->pbuf+this->size - this->pout;

    if (length < length1) {
      // if length < length1, no need to wrap around, just copy length bytes
      copy_to_user(pbuf, this->pout, length);
    }
    else {
      // we know that the buffer has wrapped, so read length1 bytes from the
      // end of the buffer and the rest of the bytes from the start
      copy_to_user(pbuf, this->pout, length1);
      copy_to_user(pbuf+length1, this->pbuf, length-length1);
    }
  }
  else {	
    // since the buffer hasn't wrapped yet, just dump bytes from the current 
    // read position (this->pout) to the user
    length = (length0 < length)?length0:length;
    copy_to_user(pbuf, this->pout, length);
  }

  // disable interrupts
  cli();
  
  // mark copied data as free in buffer and wrap around if neccesary
  this->pout += length;
  if (this->pout > this->pbuf_top) this->pout -= this->size;

  // read bytes from the parallel port
  devriport_rx(this);

  // re-enable interrupts
  sti();

  return length;
}

unsigned int devriport_poll(struct devriport* this, struct poll_table_struct* ptable)
{
	unsigned int mask = 0;
	poll_wait(this->pfile ,&this->qwait, ptable);
	if (this->pin != this->pout) mask |= POLLIN | POLLRDNORM;
	return mask;
}

void devriport_irq(struct devriport* this, int irq, struct pt_regs* regs)
{
	if (this->irqinuse) {
		devriport_rx(this);
		wake_up_interruptible(&this->qwait);
	}
}

void devriport_rx(struct devriport* this)
{
	int free;

	free = this->pin - this->pout;

	// absolute value of free... using twos complement
	if (free < 0)
		free = -(free+1);
	else
		free = this->size-(free+1);

	// read bytes in while the FIFO has bytes, read is successful, and 
	// number of bytes left free in the input buffer is greater than zero
	while(free && !(ECR_FIFO_EMPTY & inb(this->io_ext+ECR_EXT))) {

	  // if readState is RPDUMP_TIMESTAMP, put the time from the latest
	  // sync word into the data stream
	  if(this->readState != RPDUMP_TIMESTAMP) {
	    *(this->pin++) = inb(this->io_ext+FIFO_EXT);
	  }
	  else {
	    // take the timeval struct in the riport and dump it to the output
	    // buffer
	    *(this->pin++) = 
	      ((char*)&this->timeStamp)[this->numBytesThisState];
	  }

	  if (this->pin > this->pbuf_top) this->pin -= this->size;
	  free--;

	  switch(this->readState) {
	    // due to the magic of the ECP port, it seems that we are 
	    // guaranteed to be fed a header from the riegl whenever we call
	    // riport_open.  this code assumes that is true
	  case RPINIT:
	    // header length is the first 4 bytes in the header
	    // read them in and then let RPREAD_HEADER deal with the rest of 
	    // the header
	    //printk("RPINIT: numBytesThisState == %d\n", this->numBytesThisState);
	    // store the next byte
	    this->buf[(this->numBytesThisState)++] = *(this->pin - 1);

	    // after 4 bytes, we know the size of the header
	    // the next two bytes are the size of the header, this is dealt
	    // with in RPREAD_HEADER
	    if(this->numBytesThisState == 4) {
	      // get the number of bytes in the header
	      // number of bytes in the header is a 32 bit number, and includes
	      // the 4 bytes just read
	      this->bytesToRead = 
		this->buf[0] + (this->buf[1] << 8) + 
		(this->buf[2] << 16) + (this->buf[3] << 24) - 4;

	      // reset variables for RPREAD_HEADER
	      this->numBytesThisState = 0;
	      this->readState = RPREAD_HEADER;
	      //printk("RPINIT: done... Entering RPREAD_HEADER with %d bytes to read\n", this->bytesToRead);
	    }
	    break;
	  case RPREAD_HEADER:
	    // record the first two bytes, since they describe the number of
	    // bytes per read
	    if(this->numBytesThisState < 2)
	      this->buf[this->numBytesThisState++] = *(this->pin - 1);
	    else {
	      this->numBytesThisState++;
	    }

	    // after two byte reads, record the syncWord
	    if(this->numBytesThisState == 2) {
	      this->syncWord = this->buf[0] + (this->buf[1] << 8);
	    }

	    // read to the end of the header and then go to READ state
	    if(this->numBytesThisState == this->bytesToRead) {
	      //printk("RPREAD_HEADER: found the end of the header, looking for sync message\n");
	      do_gettimeofday(&this->timeStamp);

	      this->numBytesThisState = 0;
	      this->bytesToRead = 0;
	      this->readState = RPSYNC;
	    }
	    break;
	  case RPREAD:
	    // ignore all the bytes in the data packet
	    this->numBytesThisState++;
	    if(this->numBytesThisState == this->bytesToRead) {
	      //printk("RPREAD: read %d bytes\n", this->numBytesThisState);
	      this->bytesToRead = 0;
	      this->numBytesThisState = 0;
	      this->readState = RPSYNC;
	    }
	    break;
	  case RPSYNC:
	    // look for the two sync bytes

	    // just record the first byte, since we need two bytes to get the 
	    // sync
	    if(this->numBytesThisState == 0) {
	      //printk("RPSYNC: now entering state\n");
	      this->numBytesThisState++;
	      this->buf[1] = *(this->pin - 1);
	    }
	    else {
	      // push the next byte into the 2 byte queue
	      this->buf[0] = this->buf[1];
	      this->buf[1] = *(this->pin - 1);
	      this->numBytesThisState++;

	      // if the sync word matches the two bytes in storage, change the
	      // state so that timeStamp is entered into the datastream
	      if(this->syncWord == this->buf[0] + (this->buf[1] << 8)) {
		do_gettimeofday(&this->timeStamp);
		//printk("RPSYNC: found the sync word after %d slips\n", this->numBytesThisState);
		// found the sync word, so continue with READ
		this->numBytesThisState = 0;
		this->bytesToRead = this->syncWord;
		this->readState = RPDUMP_TIMESTAMP;
	      }
	    }

	    break;
	  case RPDUMP_TIMESTAMP:
	    // increment numBytesThisState to record the number of bytes
	    // passed into the data stream.  once a full timeval has been
	    // passed, move on to reading the data from the riegl
	    this->numBytesThisState++;

	    //printk("RPDUMP_TIMESTAMP: sizeof(struct timeval) == %d, so far dumped %d bytes\n", 
	    //		   sizeof(struct timeval), this->numBytesThisState);
	    if(this->numBytesThisState >= sizeof(struct timeval)) {
	      this->numBytesThisState = 0;
	      this->bytesToRead = this->syncWord;
	      this->readState = RPREAD;
	    }
	    break;
	  default:
	    this->readState = RPINIT;
	    //printk("strange behavior in riport, readState is incorrectly initialized\n");
	    break;
	  }
	}

	// if we there isn't any more space in the buffer, enable interrupts
	// otherwise disable service interrupts
	// in both cases, leave parallel port in ECP mode and disable error 
	// interrupt
	if (free)
	  // enable IRQ's
	  outb(ECR_ECP_MODE|ECR_ERRINT_DISABLED, this->io_ext+ECR_EXT);
	else
	  // disable IRQ's
	  outb(ECR_ECP_MODE|ECR_ERRINT_DISABLED|ECR_SERVICE_INTERRUPT, this->io_ext+ECR_EXT);
}

//------------------------------------------------------------------------------------//
static struct drvriport {
	int major;
	int numdevs;
	struct devriport* rgpdev[MAX_RIPORT_DEVICES];
} riport;

int drvriport_open(struct inode* pinode, struct file* pfile)
{
	int result;
	struct devriport* pdev;

	if (!(MINOR(pinode->i_rdev)<riport.numdevs)) return -ENODEV;

	pdev = riport.rgpdev[MINOR(pinode->i_rdev)];
	pdev->pinode = pinode;
	pdev->pfile = pfile;
	pfile->private_data = pdev;
	result = devriport_open(pdev);

	if (!(result<0)) MOD_INC_USE_COUNT;
	return result;
}

int drvriport_release(struct inode* pinode, struct file* pfile)
{
  //  printk("Close attempt\n");	
  devriport_release(riport.rgpdev[MINOR(pinode->i_rdev)]);

  MOD_DEC_USE_COUNT;
  return 0;
}

ssize_t drvriport_read(struct file* pfile, char* pbuf, size_t length, loff_t* ppos)
{
  return devriport_read((struct devriport*)pfile->private_data, pbuf, length);
}

unsigned int drvriport_poll(struct file* pfile, struct poll_table_struct* ptable)
{
  return devriport_poll((struct devriport*)pfile->private_data, ptable);
}

static struct file_operations drvriport_fops = {
   owner:      THIS_MODULE,
   read:      drvriport_read,
   //write:      drvriport_write,
   //ioctl:      skel_ioctl,
   open:      drvriport_open,
   release:   drvriport_release,
};
#ifdef MODULE
static int io = 0;
static int irq = 0;
static int dma = 1;
static int size = 0;
MODULE_PARM(io, "i");
MODULE_PARM(irq, "i");
MODULE_PARM(size, "i");

int init_module(void)
{
	int major = 0;
	int result;
	struct devriport* pdev;
	int n;
	
	// set default parameters
	if (0==io) io = RIPORT_IO;
	if (0==irq) irq = RIPORT_IRQ;	
	if (0==size) size = RIPORT_SIZE;
		
	if ((result = register_chrdev(major,"riport",&drvriport_fops))<0)
		goto fail_register_chrdev;

	// TODO: here is the place to add more riport devices
	riport.major = result;	
	riport.numdevs = 0;
	pdev = devriport_init(riport.major, riport.numdevs, io,irq,dma,size,&result);	
	if(!pdev) goto fail_dev;
	riport.rgpdev[riport.numdevs++] = pdev;
	
	return 0;

	fail_dev:
		for (n=0; n<riport.numdevs; n++)
			devriport_cleanup(riport.rgpdev[n]);
		unregister_chrdev(riport.major,"riport");	
	fail_register_chrdev:
		return result;	
}

void cleanup_module(void)
{
	int n;
	for (n=0; n<riport.numdevs; n++)
		devriport_cleanup(riport.rgpdev[n]);
	unregister_chrdev(riport.major,"riport");
}
#endif

